<?php

namespace App\Http\Controllers;

use App\Models\ReportComponent;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ReportComponentController extends Controller
{
    public function index()
    {
        $components = ReportComponent::with('user')->get();
        return response()->json($components);
    }
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'remark' => 'nullable|string',
            'user_id' => 'required|exists:users,id',
        ]);
    
        // Generate a unique code (e.g., RC-20250410-XXXX)
        $code = $this->generateUniqueCode();
    
        $component = ReportComponent::create([
            'name' => $request->name,
            'code' => $code,
            'remark' => $request->remark,
            'status' => $request->status ?? 1,
            'user_id' => $request->user_id
        ]);
    
        return response()->json($component, 201);
    }
    
    /**
     * Generate a unique report component code
     */
    private function generateUniqueCode()
    {
        do {
            $code = 'RC-' . now()->format('Ymd') . '-' . strtoupper(Str::random(4));
        } while (\App\Models\ReportComponent::where('code', $code)->exists());
    
        return $code;
    }
    

    public function show(ReportComponent $reportComponent)
    {
        return response()->json($reportComponent->load('user'));
    }

    public function update(Request $request, ReportComponent $reportComponent)
    {
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'code' => 'sometimes|string|max:50|unique:report_components,code,'.$reportComponent->id,
            'remark' => 'nullable|string',
            'status' => 'sometimes|boolean',
            'user_id' => 'sometimes|exists:users,id',
        ]);

        $reportComponent->update($request->all());

        return response()->json($reportComponent);
    }

    public function destroy(ReportComponent $reportComponent)
    {
        $reportComponent->delete();
        return response()->json(null, 204);
    }
}