<?php
namespace App\Http\Controllers;
use App\Models\ExpenseType;
use Illuminate\Support\Str;
use Illuminate\Http\Request;

class ExpenseTypeController extends Controller
{
  
    public function index()
    {
        $expenseTypes = ExpenseType::with(['component', 'user'])->get();
        return response()->json($expenseTypes);
    }

    public function store(Request $request)
    {
        $request->validate([
            'component_id' => 'required|exists:report_components,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'user_id' => 'required|exists:users,id',
        ]);
        $code = $this->generateUniqueCode();
        $expenseType = ExpenseType::create([
            'component_id' => $request->component_id,
            'name' => $request->name,
            'code' =>$code,
            'description' => $request->description,
            'status' => $request->status ?? 1,
            'user_id' => $request->user_id
        ]);

        return response()->json($expenseType, 201);
    }

    private function generateUniqueCode()
    {
        do {
            $code = 'ET-' . now()->format('Ymd') . '-' . strtoupper(Str::random(4));
        } while (\App\Models\ExpenseType::where('code', $code)->exists());
    
        return $code;
    }
    

    public function show(ExpenseType $expenseType)
    {
        return response()->json($expenseType->load(['component', 'user']));
    }

    public function update(Request $request, ExpenseType $expenseType)
    {
        $request->validate([
            'component_id' => 'sometimes|exists:report_components,id',
            'name' => 'sometimes|string|max:255',
            'code' => 'sometimes|string|max:50|unique:expenses_types,code,'.$expenseType->id,
            'description' => 'nullable|string',
            'status' => 'sometimes|boolean',
            'user_id' => 'sometimes|exists:users,id',
        ]);

        $expenseType->update($request->all());

        return response()->json($expenseType);
    }

    public function destroy(ExpenseType $expenseType)
    {
        $expenseType->delete();
        return response()->json(null, 204);
    }
}