<?php

namespace App\Http\Controllers;

use App\Models\Asset;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class AssetController extends Controller
{
    public function index()
    {
        $assets = Asset::with(['component', 'user'])->get();
        return response()->json($assets);
    }

    public function store(Request $request)
    {
        $request->validate([
            'component_id' => 'required|exists:report_components,id',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'user_id' => 'required|exists:users,id',
        ]);
        $code = $this->generateUniqueCode();
        $asset = Asset::create([
            'component_id' => $request->component_id,
            'name' => $request->name,
            'code' => $code,
            'description' => $request->description,
            'status' => $request->status ?? 1,
            'user_id' => $request->user_id
        ]);

        return response()->json($asset, 201);
    }

    private function generateUniqueCode()
    {
        do {
            $code = 'AC-' . now()->format('Ymd') . '-' . strtoupper(Str::random(4));
        } while (\App\Models\Asset::where('code', $code)->exists());
    
        return $code;
    }
    

    public function show(Asset $asset)
    {
        return response()->json($asset->load(['component', 'user']));
    }

    public function update(Request $request, Asset $asset)
    {
        $request->validate([
            'component_id' => 'sometimes|exists:report_components,id',
            'name' => 'sometimes|string|max:255',
            'code' => 'sometimes|string|max:50|unique:assets,code,'.$asset->id,
            'description' => 'nullable|string',
            'status' => 'sometimes|boolean',
            'user_id' => 'sometimes|exists:users,id',
        ]);

        $asset->update($request->all());

        return response()->json($asset);
    }

    public function destroy(Asset $asset)
    {
        $asset->delete();
        return response()->json(null, 204);
    }
}